/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.blobcache.common;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.RefCountingListener;
import org.elasticsearch.blobcache.common.ByteRange;
import org.elasticsearch.blobcache.common.ProgressListenableActionFuture;
import org.elasticsearch.core.Assertions;
import org.elasticsearch.core.Nullable;

public class SparseFileTracker {
    private static final Comparator<Range> RANGE_START_COMPARATOR = Comparator.comparingLong(r -> r.start);
    private final TreeSet<Range> ranges = new TreeSet<Range>(RANGE_START_COMPARATOR);
    private volatile long complete = 0L;
    private final Object mutex = new Object();
    private final String description;
    private final long length;
    private final long initialLength;

    public SparseFileTracker(String description, long length) {
        this(description, length, Collections.emptySortedSet());
    }

    public SparseFileTracker(String description, long length, SortedSet<ByteRange> ranges) {
        this.description = description;
        this.length = length;
        if (length < 0L) {
            throw new IllegalArgumentException("Length [" + length + "] must be equal to or greater than 0 for [" + description + "]");
        }
        this.initialLength = ranges.isEmpty() ? 0L : this.addInitialRanges(length, ranges);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long addInitialRanges(long length, SortedSet<ByteRange> ranges) {
        long initialLength = 0L;
        Object object = this.mutex;
        synchronized (object) {
            Range previous = null;
            for (ByteRange next : ranges) {
                if (next.isEmpty()) {
                    throw new IllegalArgumentException("Range " + next + " cannot be empty");
                }
                if (length < next.end()) {
                    throw new IllegalArgumentException("Range " + next + " is exceeding maximum length [" + length + "]");
                }
                Range range = new Range(next);
                if (previous != null && range.start <= previous.end) {
                    throw new IllegalArgumentException("Range " + range + " is overlapping a previous range " + previous);
                }
                boolean added = this.ranges.add(range);
                assert (added) : range + " already exist in " + this.ranges;
                previous = range;
                initialLength += range.end - range.start;
            }
            assert (this.invariant());
        }
        return initialLength;
    }

    public long getLength() {
        return this.length;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public SortedSet<ByteRange> getCompletedRanges() {
        TreeSet<ByteRange> completedRanges = null;
        Object object = this.mutex;
        synchronized (object) {
            assert (this.invariant());
            for (Range range : this.ranges) {
                if (range.isPending()) continue;
                if (completedRanges == null) {
                    completedRanges = new TreeSet<ByteRange>();
                }
                completedRanges.add(ByteRange.of(range.start, range.end));
            }
        }
        return completedRanges == null ? Collections.emptySortedSet() : completedRanges;
    }

    public long getInitialLength() {
        return this.initialLength;
    }

    private long computeLengthOfRanges() {
        assert (Thread.holdsLock(this.mutex)) : "sum of length of the ranges must be computed under mutex";
        return this.ranges.stream().mapToLong(range -> range.end - range.start).sum();
    }

    public List<Gap> waitForRange(ByteRange range, ByteRange subRange, ActionListener<Void> listener) {
        if (this.length < range.end()) {
            throw new IllegalArgumentException("invalid range [" + range + ", length=" + this.length + "]");
        }
        if (this.length < subRange.end()) {
            throw new IllegalArgumentException("invalid range to listen to [" + subRange + ", length=" + this.length + "]");
        }
        if (!subRange.isSubRangeOf(range)) {
            throw new IllegalArgumentException("unable to listen to range [start=" + subRange.start() + ", end=" + subRange.end() + "] when range is [start=" + range.start() + ", end=" + range.end() + ", length=" + this.length + "]");
        }
        if (this.complete >= range.end()) {
            listener.onResponse(null);
            return List.of();
        }
        return this.doWaitForRange(range, subRange, listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<Gap> doWaitForRange(ByteRange range, ByteRange subRange, ActionListener<Void> listener) {
        ActionListener<Void> wrappedListener = this.wrapWithAssertions(listener);
        ArrayList<Gap> gaps = new ArrayList<Gap>();
        ArrayList<Range> pendingRanges = new ArrayList<Range>();
        Range targetRange = new Range(range);
        Object object = this.mutex;
        synchronized (object) {
            this.determineStartingRange(range, pendingRanges, targetRange);
            while (targetRange.start < range.end()) {
                assert (0L <= targetRange.start) : targetRange;
                assert (this.invariant());
                Range firstExistingRange = this.ranges.ceiling(targetRange);
                if (firstExistingRange == null) {
                    Range newPendingRange = new Range(targetRange.start, range.end(), new ProgressListenableActionFuture(targetRange.start, range.end()));
                    this.ranges.add(newPendingRange);
                    pendingRanges.add(newPendingRange);
                    gaps.add(new Gap(newPendingRange));
                    targetRange.start = range.end();
                    continue;
                }
                assert (targetRange.start <= firstExistingRange.start) : targetRange + " vs " + firstExistingRange;
                if (targetRange.start == firstExistingRange.start) {
                    if (firstExistingRange.isPending()) {
                        pendingRanges.add(firstExistingRange);
                    }
                    targetRange.start = Math.min(range.end(), firstExistingRange.end);
                    continue;
                }
                long newPendingRangeEnd = Math.min(range.end(), firstExistingRange.start);
                Range newPendingRange = new Range(targetRange.start, newPendingRangeEnd, new ProgressListenableActionFuture(targetRange.start, newPendingRangeEnd));
                this.ranges.add(newPendingRange);
                pendingRanges.add(newPendingRange);
                gaps.add(new Gap(newPendingRange));
                targetRange.start = newPendingRange.end;
            }
            assert (targetRange.start == targetRange.end) : targetRange;
            assert (targetRange.start == range.end()) : targetRange;
            assert (this.invariant());
            assert (this.ranges.containsAll(pendingRanges)) : this.ranges + " vs " + pendingRanges;
            assert (pendingRanges.stream().allMatch(Range::isPending)) : pendingRanges;
            assert (pendingRanges.size() != 1 || gaps.size() <= 1) : gaps;
        }
        if (!range.equals(subRange)) {
            pendingRanges.removeIf(pendingRange -> !(pendingRange.start < subRange.end() && subRange.start() < pendingRange.end));
            pendingRanges.sort(RANGE_START_COMPARATOR);
        }
        SparseFileTracker.subscribeToCompletionListeners(pendingRanges, subRange.end(), wrappedListener);
        return Collections.unmodifiableList(gaps);
    }

    private void determineStartingRange(ByteRange range, List<Range> pendingRanges, Range targetRange) {
        assert (this.invariant());
        Range lastEarlierRange = this.ranges.lower(targetRange);
        if (lastEarlierRange != null && range.start() < lastEarlierRange.end) {
            if (lastEarlierRange.isPending()) {
                pendingRanges.add(lastEarlierRange);
            }
            targetRange.start = Math.min(range.end(), lastEarlierRange.end);
        }
    }

    public boolean checkAvailable(long upTo) {
        assert (upTo <= this.length) : "tried to check availability up to [" + upTo + "] but length is only [" + this.length + "]";
        return this.complete >= upTo;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean waitForRangeIfPending(ByteRange range, ActionListener<Void> listener) {
        if (this.length < range.end()) {
            throw new IllegalArgumentException("invalid range [" + range + ", length=" + this.length + "]");
        }
        ActionListener<Void> wrappedListener = this.wrapWithAssertions(listener);
        ArrayList<Range> pendingRanges = new ArrayList<Range>();
        Range targetRange = new Range(range);
        Object object = this.mutex;
        synchronized (object) {
            this.determineStartingRange(range, pendingRanges, targetRange);
            while (targetRange.start < range.end()) {
                assert (0L <= targetRange.start) : targetRange;
                assert (this.invariant());
                Range firstExistingRange = this.ranges.ceiling(targetRange);
                if (firstExistingRange == null) {
                    return false;
                }
                assert (targetRange.start <= firstExistingRange.start) : targetRange + " vs " + firstExistingRange;
                if (targetRange.start == firstExistingRange.start) {
                    if (firstExistingRange.isPending()) {
                        pendingRanges.add(firstExistingRange);
                    }
                    targetRange.start = Math.min(range.end(), firstExistingRange.end);
                    continue;
                }
                return false;
            }
            assert (targetRange.start == targetRange.end) : targetRange;
            assert (targetRange.start == range.end()) : targetRange;
            assert (this.invariant());
        }
        SparseFileTracker.subscribeToCompletionListeners(pendingRanges, range.end(), wrappedListener);
        return true;
    }

    private static void subscribeToCompletionListeners(List<Range> requiredRanges, long rangeEnd, ActionListener<Void> listener) {
        switch (requiredRanges.size()) {
            case 0: {
                listener.onResponse(null);
                break;
            }
            case 1: {
                Range requiredRange = requiredRanges.get(0);
                requiredRange.completionListener.addListener((ActionListener<Long>)listener.map(progress -> null), Math.min(requiredRange.completionListener.end, rangeEnd));
                break;
            }
            default: {
                try (RefCountingListener listeners = new RefCountingListener(listener);){
                    for (Range range : requiredRanges) {
                        range.completionListener.addListener((ActionListener<Long>)listeners.acquire(l -> {}), Math.min(range.completionListener.end, rangeEnd));
                    }
                    break;
                }
            }
        }
    }

    private ActionListener<Void> wrapWithAssertions(ActionListener<Void> listener) {
        if (Assertions.ENABLED) {
            return ActionListener.runAfter(listener, () -> {
                assert (!Thread.holdsLock(this.mutex)) : "mutex unexpectedly held in listener";
            });
        }
        return listener;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public ByteRange getAbsentRangeWithin(ByteRange range) {
        Object object = this.mutex;
        synchronized (object) {
            long start = range.start();
            Range lastStartRange = this.ranges.floor(new Range(start, start, null));
            long resultStart = lastStartRange == null ? start : (lastStartRange.end < start ? start : (lastStartRange.isPending() ? start : lastStartRange.end));
            assert (resultStart >= start);
            long end = range.end();
            Range lastEndRange = this.ranges.lower(new Range(end, end, null));
            long resultEnd = lastEndRange == null ? end : (lastEndRange.end < end ? end : (lastEndRange.isPending() ? end : lastEndRange.start));
            assert (resultEnd <= end);
            return resultStart < resultEnd ? ByteRange.of(resultStart, resultEnd) : null;
        }
    }

    private boolean assertPendingRangeExists(Range range) {
        assert (Thread.holdsLock(this.mutex));
        SortedSet<Range> existingRanges = this.ranges.tailSet(range);
        assert (!existingRanges.isEmpty());
        Range existingRange = existingRanges.first();
        assert (existingRange == range);
        assert (existingRange.isPending());
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onGapSuccess(Range gapRange) {
        Object object = this.mutex;
        synchronized (object) {
            boolean mergeWithNext;
            Range nextRange;
            Range prevRange;
            assert (this.invariant());
            assert (this.assertPendingRangeExists(gapRange));
            this.ranges.remove(gapRange);
            SortedSet<Range> prevRanges = this.ranges.headSet(gapRange);
            Range range = prevRange = prevRanges.isEmpty() ? null : prevRanges.last();
            assert (prevRange == null || prevRange.end <= gapRange.start) : prevRange + " vs " + gapRange;
            boolean mergeWithPrev = prevRange != null && !prevRange.isPending() && prevRange.end == gapRange.start;
            SortedSet<Range> nextRanges = this.ranges.tailSet(gapRange);
            Range range2 = nextRange = nextRanges.isEmpty() ? null : nextRanges.first();
            assert (nextRange == null || gapRange.end <= nextRange.start) : gapRange + " vs " + nextRange;
            boolean bl = mergeWithNext = nextRange != null && !nextRange.isPending() && gapRange.end == nextRange.start;
            if (mergeWithPrev && mergeWithNext) {
                assert (!prevRange.isPending()) : prevRange;
                assert (!nextRange.isPending()) : nextRange;
                assert (prevRange.end == gapRange.start) : prevRange + " vs " + gapRange;
                assert (gapRange.end == nextRange.start) : gapRange + " vs " + nextRange;
                prevRange.end = nextRange.end;
                this.ranges.remove(nextRange);
                this.maybeUpdateCompletePointer(prevRange);
            } else if (mergeWithPrev) {
                assert (!prevRange.isPending()) : prevRange;
                assert (prevRange.end == gapRange.start) : prevRange + " vs " + gapRange;
                prevRange.end = gapRange.end;
                this.maybeUpdateCompletePointer(prevRange);
            } else if (mergeWithNext) {
                assert (!nextRange.isPending()) : nextRange;
                assert (gapRange.end == nextRange.start) : gapRange + " vs " + nextRange;
                nextRange.start = gapRange.start;
                this.maybeUpdateCompletePointer(nextRange);
            } else {
                this.maybeUpdateCompletePointer(gapRange);
                this.ranges.add(new Range(gapRange.start, gapRange.end, null));
            }
            assert (this.invariant());
        }
        gapRange.completionListener.onResponse(gapRange.end);
    }

    private void maybeUpdateCompletePointer(Range gapRange) {
        assert (Thread.holdsLock(this.mutex));
        if (gapRange.start == 0L) {
            assert (this.complete <= gapRange.end);
            this.complete = gapRange.end;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean assertGapRangePending(Range gapRange) {
        Object object = this.mutex;
        synchronized (object) {
            assert (this.invariant());
            assert (this.assertPendingRangeExists(gapRange));
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onGapFailure(Range gapRange, Exception e) {
        Object object = this.mutex;
        synchronized (object) {
            assert (this.invariant());
            assert (this.assertPendingRangeExists(gapRange));
            boolean removed = this.ranges.remove(gapRange);
            assert (removed) : gapRange + " not found";
            assert (this.invariant());
        }
        gapRange.completionListener.onFailure(e);
    }

    private boolean invariant() {
        assert (Thread.holdsLock(this.mutex));
        long lengthOfRanges = 0L;
        Range previousRange = null;
        for (Range range : this.ranges) {
            if (previousRange != null) {
                assert (range.start < range.end) : range;
                assert (previousRange.end <= range.start) : previousRange + " vs " + range;
                assert (previousRange.isPending() || range.isPending() || previousRange.end < range.start) : previousRange + " vs " + range;
            }
            assert (range.end <= this.length);
            lengthOfRanges += range.end - range.start;
            previousRange = range;
        }
        assert (this.computeLengthOfRanges() <= this.length);
        assert (this.computeLengthOfRanges() == lengthOfRanges);
        return true;
    }

    public String toString() {
        return "SparseFileTracker{description=" + this.description + ", length=" + this.length + ", complete=" + this.complete + "}";
    }

    private static class Range {
        long start;
        long end;
        @Nullable
        final ProgressListenableActionFuture completionListener;

        Range(ByteRange range) {
            this(range.start(), range.end(), null);
        }

        Range(long start, long end, @Nullable ProgressListenableActionFuture completionListener) {
            assert (start <= end) : start + "-" + end;
            this.start = start;
            this.end = end;
            this.completionListener = completionListener;
        }

        boolean isPending() {
            return this.completionListener != null;
        }

        public String toString() {
            return "[" + this.start + "-" + this.end + (this.isPending() ? ", pending]" : "]");
        }
    }

    public class Gap {
        public final Range range;

        Gap(Range range) {
            assert (range.start < range.end) : range.start + "-" + range.end;
            this.range = range;
        }

        public long start() {
            return this.range.start;
        }

        public long end() {
            return this.range.end;
        }

        public void onCompletion() {
            SparseFileTracker.this.onGapSuccess(this.range);
        }

        public void onProgress(long value) {
            assert (SparseFileTracker.this.assertGapRangePending(this.range));
            this.range.completionListener.onProgress(value);
        }

        public void onFailure(Exception e) {
            SparseFileTracker.this.onGapFailure(this.range, e);
        }

        public String toString() {
            return SparseFileTracker.this.toString() + " " + this.range;
        }
    }
}

