/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.gcs;

import com.google.api.gax.paging.Page;
import com.google.cloud.BaseServiceException;
import com.google.cloud.BatchResult;
import com.google.cloud.ReadChannel;
import com.google.cloud.WriteChannel;
import com.google.cloud.storage.Blob;
import com.google.cloud.storage.BlobId;
import com.google.cloud.storage.BlobInfo;
import com.google.cloud.storage.Storage;
import com.google.cloud.storage.StorageBatch;
import com.google.cloud.storage.StorageException;
import java.io.ByteArrayInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileAlreadyExistsException;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.BytesRef;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStore;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.common.blobstore.OptionalBytesReference;
import org.elasticsearch.common.blobstore.support.BlobContainerUtils;
import org.elasticsearch.common.blobstore.support.BlobMetadata;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.io.stream.ReleasableBytesStreamOutput;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.BigArrays;
import org.elasticsearch.core.CheckedConsumer;
import org.elasticsearch.core.CheckedRunnable;
import org.elasticsearch.core.Streams;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageBlobContainer;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageOperationsStats;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageRetryingInputStream;
import org.elasticsearch.repositories.gcs.GoogleCloudStorageService;
import org.elasticsearch.repositories.gcs.SocketAccess;
import org.elasticsearch.rest.RestStatus;

class GoogleCloudStorageBlobStore
implements BlobStore {
    private static final Logger logger = LogManager.getLogger(GoogleCloudStorageBlobStore.class);
    public static final int LARGE_BLOB_THRESHOLD_BYTE_SIZE;
    public static final int MAX_DELETES_PER_BATCH = 1000;
    private final String bucketName;
    private final String clientName;
    private final String repositoryName;
    private final GoogleCloudStorageService storageService;
    private final GoogleCloudStorageOperationsStats stats;
    private final int bufferSize;
    private final BigArrays bigArrays;
    private static final Storage.BlobWriteOption[] NO_OVERWRITE_NO_MD5;
    private static final Storage.BlobWriteOption[] OVERWRITE_NO_MD5;
    private static final Storage.BlobWriteOption[] NO_OVERWRITE_CHECK_MD5;
    private static final Storage.BlobWriteOption[] OVERWRITE_CHECK_MD5;

    GoogleCloudStorageBlobStore(String bucketName, String clientName, String repositoryName, GoogleCloudStorageService storageService, BigArrays bigArrays, int bufferSize) {
        this.bucketName = bucketName;
        this.clientName = clientName;
        this.repositoryName = repositoryName;
        this.storageService = storageService;
        this.bigArrays = bigArrays;
        this.stats = new GoogleCloudStorageOperationsStats(bucketName);
        this.bufferSize = bufferSize;
    }

    private Storage client() throws IOException {
        return this.storageService.client(this.clientName, this.repositoryName, this.stats);
    }

    public BlobContainer blobContainer(BlobPath path) {
        return new GoogleCloudStorageBlobContainer(path, this);
    }

    public void close() {
        this.storageService.closeRepositoryClient(this.repositoryName);
    }

    Map<String, BlobMetadata> listBlobs(String path) throws IOException {
        return this.listBlobsByPrefix(path, "");
    }

    Map<String, BlobMetadata> listBlobsByPrefix(String path, String prefix) throws IOException {
        String pathPrefix = GoogleCloudStorageBlobStore.buildKey(path, prefix);
        HashMap mapBuilder = new HashMap();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathPrefix)}).iterateAll().forEach(blob -> {
            assert (blob.getName().startsWith(path));
            if (!blob.isDirectory()) {
                String suffixName = blob.getName().substring(path.length());
                mapBuilder.put(suffixName, new BlobMetadata(suffixName, blob.getSize().longValue()));
            }
        })));
        return Map.copyOf(mapBuilder);
    }

    Map<String, BlobContainer> listChildren(BlobPath path) throws IOException {
        String pathStr = path.buildAsString();
        HashMap mapBuilder = new HashMap();
        SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.currentDirectory(), Storage.BlobListOption.prefix((String)pathStr)}).iterateAll().forEach(blob -> {
            if (blob.isDirectory()) {
                assert (blob.getName().startsWith(pathStr));
                assert (blob.getName().endsWith("/"));
                String suffixName = blob.getName().substring(pathStr.length(), blob.getName().length() - 1);
                if (!suffixName.isEmpty()) {
                    mapBuilder.put(suffixName, new GoogleCloudStorageBlobContainer(path.add(suffixName), this));
                }
            }
        })));
        return Map.copyOf(mapBuilder);
    }

    boolean blobExists(String blobName) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        Blob blob = SocketAccess.doPrivilegedIOException(() -> this.client().get(blobId));
        return blob != null;
    }

    InputStream readBlob(String blobName) throws IOException {
        return new GoogleCloudStorageRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName));
    }

    InputStream readBlob(String blobName, long position, long length) throws IOException {
        if (position < 0L) {
            throw new IllegalArgumentException("position must be non-negative");
        }
        if (length < 0L) {
            throw new IllegalArgumentException("length must be non-negative");
        }
        if (length == 0L) {
            return new ByteArrayInputStream(new byte[0]);
        }
        return new GoogleCloudStorageRetryingInputStream(this.client(), BlobId.of((String)this.bucketName, (String)blobName), position, Math.addExact(position, length - 1L));
    }

    void writeBlob(String blobName, BytesReference bytes, boolean failIfAlreadyExists) throws IOException {
        if ((long)bytes.length() > this.getLargeBlobThresholdInBytes()) {
            String md5 = Base64.getEncoder().encodeToString(MessageDigests.digest((BytesReference)bytes, (MessageDigest)MessageDigests.md5()));
            this.writeBlobResumable(BlobInfo.newBuilder((String)this.bucketName, (String)blobName).setMd5(md5).build(), (InputStream)bytes.streamInput(), bytes.length(), failIfAlreadyExists);
        } else {
            BlobInfo blobInfo = BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build();
            if (bytes.hasArray()) {
                this.writeBlobMultipart(blobInfo, bytes.array(), bytes.arrayOffset(), bytes.length(), failIfAlreadyExists);
            } else {
                this.writeBlob((InputStream)bytes.streamInput(), bytes.length(), failIfAlreadyExists, blobInfo);
            }
        }
    }

    void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        this.writeBlob(inputStream, blobSize, failIfAlreadyExists, BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build());
    }

    private void writeBlob(InputStream inputStream, long blobSize, boolean failIfAlreadyExists, BlobInfo blobInfo) throws IOException {
        if (blobSize > this.getLargeBlobThresholdInBytes()) {
            this.writeBlobResumable(blobInfo, inputStream, blobSize, failIfAlreadyExists);
        } else {
            byte[] buffer = new byte[Math.toIntExact(blobSize)];
            Streams.readFully((InputStream)inputStream, (byte[])buffer);
            this.writeBlobMultipart(blobInfo, buffer, 0, Math.toIntExact(blobSize), failIfAlreadyExists);
        }
    }

    long getLargeBlobThresholdInBytes() {
        return LARGE_BLOB_THRESHOLD_BYTE_SIZE;
    }

    void writeBlob(String blobName, boolean failIfAlreadyExists, CheckedConsumer<OutputStream, IOException> writer) throws IOException {
        final BlobInfo blobInfo = BlobInfo.newBuilder((String)this.bucketName, (String)blobName).build();
        final Storage.BlobWriteOption[] writeOptions = failIfAlreadyExists ? NO_OVERWRITE_NO_MD5 : OVERWRITE_NO_MD5;
        StorageException storageException = null;
        for (int retry = 0; retry < 3; ++retry) {
            try (final ReleasableBytesStreamOutput buffer = new ReleasableBytesStreamOutput(this.bigArrays);){
                final AtomicReference channelRef = new AtomicReference();
                writer.accept((Object)new OutputStream(){
                    private OutputStream resumableStream;

                    @Override
                    public void write(int b) throws IOException {
                        if (this.resumableStream != null) {
                            this.resumableStream.write(b);
                        } else if ((long)(buffer.size() + 1) > GoogleCloudStorageBlobStore.this.getLargeBlobThresholdInBytes()) {
                            this.initResumableStream();
                            this.resumableStream.write(b);
                        } else {
                            buffer.write(b);
                        }
                    }

                    @Override
                    public void write(byte[] b, int off, int len) throws IOException {
                        if (this.resumableStream != null) {
                            this.resumableStream.write(b, off, len);
                        } else if ((long)(buffer.size() + len) > GoogleCloudStorageBlobStore.this.getLargeBlobThresholdInBytes()) {
                            this.initResumableStream();
                            this.resumableStream.write(b, off, len);
                        } else {
                            buffer.write(b, off, len);
                        }
                    }

                    private void initResumableStream() throws IOException {
                        WriteChannel writeChannel = SocketAccess.doPrivilegedIOException(() -> GoogleCloudStorageBlobStore.this.client().writer(blobInfo, writeOptions));
                        channelRef.set(writeChannel);
                        this.resumableStream = new FilterOutputStream(Channels.newOutputStream(new WritableBlobChannel(writeChannel))){

                            @Override
                            public void write(byte[] b, int off, int len) throws IOException {
                                int toWrite;
                                for (int written = 0; written < len; written += toWrite) {
                                    toWrite = Math.min(len - written, 0xF00000);
                                    this.out.write(b, off + written, toWrite);
                                }
                            }
                        };
                        buffer.bytes().writeTo(this.resumableStream);
                        buffer.close();
                    }
                });
                WritableByteChannel writeChannel = (WritableByteChannel)channelRef.get();
                if (writeChannel != null) {
                    SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)writeChannel::close));
                    this.stats.trackPutOperation();
                } else {
                    this.writeBlob(blobName, buffer.bytes(), failIfAlreadyExists);
                }
                return;
            }
            catch (StorageException se) {
                int errorCode = se.getCode();
                if (errorCode != 410) {
                    if (failIfAlreadyExists && errorCode == 412) {
                        throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
                    }
                    if (storageException != null) {
                        se.addSuppressed(storageException);
                    }
                    throw se;
                }
                logger.warn(() -> Strings.format((String)"Retrying broken resumable upload session for blob %s", (Object[])new Object[]{blobInfo}), (Throwable)se);
                storageException = (StorageException)ExceptionsHelper.useOrSuppress(storageException, (Throwable)se);
                continue;
            }
        }
        assert (storageException != null);
        throw storageException;
    }

    private void writeBlobResumable(BlobInfo blobInfo, InputStream inputStream, long size, boolean failIfAlreadyExists) throws IOException {
        assert (inputStream.markSupported());
        inputStream.mark(Integer.MAX_VALUE);
        byte[] buffer = new byte[size < (long)this.bufferSize ? Math.toIntExact(size) : this.bufferSize];
        StorageException storageException = null;
        Storage.BlobWriteOption[] writeOptions = blobInfo.getMd5() == null ? (failIfAlreadyExists ? NO_OVERWRITE_NO_MD5 : OVERWRITE_NO_MD5) : (failIfAlreadyExists ? NO_OVERWRITE_CHECK_MD5 : OVERWRITE_CHECK_MD5);
        for (int retry = 0; retry < 3; ++retry) {
            try {
                WriteChannel writeChannel = SocketAccess.doPrivilegedIOException(() -> this.client().writer(blobInfo, writeOptions));
                Streams.copy((InputStream)inputStream, (OutputStream)Channels.newOutputStream(new WritableBlobChannel(writeChannel)), (byte[])buffer);
                SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> writeChannel.close()));
                this.stats.trackPutOperation();
                return;
            }
            catch (StorageException se) {
                int errorCode = se.getCode();
                if (errorCode != 410) {
                    if (failIfAlreadyExists && errorCode == 412) {
                        throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
                    }
                    if (storageException != null) {
                        se.addSuppressed(storageException);
                    }
                    throw se;
                }
                logger.warn(() -> Strings.format((String)"Retrying broken resumable upload session for blob %s", (Object[])new Object[]{blobInfo}), (Throwable)se);
                storageException = (StorageException)ExceptionsHelper.useOrSuppress(storageException, (Throwable)se);
                inputStream.reset();
                continue;
            }
        }
        assert (storageException != null);
        throw storageException;
    }

    private void writeBlobMultipart(BlobInfo blobInfo, byte[] buffer, int offset, int blobSize, boolean failIfAlreadyExists) throws IOException {
        assert ((long)blobSize <= this.getLargeBlobThresholdInBytes()) : "large blob uploads should use the resumable upload method";
        try {
            Storage.BlobTargetOption[] blobTargetOptionArray;
            if (failIfAlreadyExists) {
                Storage.BlobTargetOption[] blobTargetOptionArray2 = new Storage.BlobTargetOption[1];
                blobTargetOptionArray = blobTargetOptionArray2;
                blobTargetOptionArray2[0] = Storage.BlobTargetOption.doesNotExist();
            } else {
                blobTargetOptionArray = new Storage.BlobTargetOption[]{};
            }
            Storage.BlobTargetOption[] targetOptions = blobTargetOptionArray;
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().create(blobInfo, buffer, offset, blobSize, targetOptions)));
            this.stats.trackPostOperation();
        }
        catch (StorageException se) {
            if (failIfAlreadyExists && se.getCode() == 412) {
                throw new FileAlreadyExistsException(blobInfo.getBlobId().getName(), null, se.getMessage());
            }
            throw se;
        }
    }

    DeleteResult deleteDirectory(OperationPurpose purpose, String pathStr) throws IOException {
        return SocketAccess.doPrivilegedIOException(() -> {
            DeleteResult deleteResult = DeleteResult.ZERO;
            Page page = this.client().list(this.bucketName, new Storage.BlobListOption[]{Storage.BlobListOption.prefix((String)pathStr)});
            do {
                final AtomicLong blobsDeleted = new AtomicLong(0L);
                final AtomicLong bytesDeleted = new AtomicLong(0L);
                final Iterator blobs = page.getValues().iterator();
                this.deleteBlobsIgnoringIfNotExists(purpose, new Iterator<String>(){

                    @Override
                    public boolean hasNext() {
                        return blobs.hasNext();
                    }

                    @Override
                    public String next() {
                        Blob next = (Blob)blobs.next();
                        blobsDeleted.incrementAndGet();
                        bytesDeleted.addAndGet(next.getSize());
                        return next.getName();
                    }
                });
                deleteResult = deleteResult.add(blobsDeleted.get(), bytesDeleted.get());
            } while ((page = page.getNextPage()) != null);
            return deleteResult;
        });
    }

    public void deleteBlobsIgnoringIfNotExists(OperationPurpose purpose, final Iterator<String> blobNames) throws IOException {
        if (!blobNames.hasNext()) {
            return;
        }
        Iterator<BlobId> blobIdsToDelete = new Iterator<BlobId>(){

            @Override
            public boolean hasNext() {
                return blobNames.hasNext();
            }

            @Override
            public BlobId next() {
                return BlobId.of((String)GoogleCloudStorageBlobStore.this.bucketName, (String)((String)blobNames.next()));
            }
        };
        final List failedBlobs = Collections.synchronizedList(new ArrayList());
        try {
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> {
                StorageException exception;
                final AtomicReference ioe = new AtomicReference();
                StorageBatch batch = this.client().batch();
                int pendingDeletesInBatch = 0;
                while (blobIdsToDelete.hasNext()) {
                    final BlobId blob = (BlobId)blobIdsToDelete.next();
                    batch.delete(blob, new Storage.BlobSourceOption[0]).notify((BatchResult.Callback)new BatchResult.Callback<Boolean, StorageException>(){

                        public void success(Boolean result) {
                        }

                        public void error(StorageException exception) {
                            if (exception.getCode() != 404) {
                                if (failedBlobs.size() < 10) {
                                    failedBlobs.add(blob);
                                }
                                if (!ioe.compareAndSet(null, exception)) {
                                    ((StorageException)ioe.get()).addSuppressed((Throwable)exception);
                                }
                            }
                        }
                    });
                    if (++pendingDeletesInBatch % 1000 != 0) continue;
                    batch.submit();
                    batch = this.client().batch();
                    pendingDeletesInBatch = 0;
                }
                if (pendingDeletesInBatch > 0) {
                    batch.submit();
                }
                if ((exception = (StorageException)((Object)ioe.get())) != null) {
                    throw exception;
                }
            }));
        }
        catch (Exception e) {
            throw new IOException("Exception when deleting blobs " + failedBlobs, e);
        }
        assert (failedBlobs.isEmpty());
    }

    private static String buildKey(String keyPath, String s) {
        assert (s != null);
        return keyPath + s;
    }

    public Map<String, Long> stats() {
        return this.stats.toMap();
    }

    /*
     * Enabled aggressive exception aggregation
     */
    OptionalBytesReference getRegister(String blobName, String container, String key) throws IOException {
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        try (ReadChannel readChannel = SocketAccess.doPrivilegedIOException(() -> this.client().reader(blobId, new Storage.BlobSourceOption[0]));){
            OptionalBytesReference optionalBytesReference;
            try (PrivilegedReadChannelStream stream = new PrivilegedReadChannelStream((ReadableByteChannel)readChannel);){
                optionalBytesReference = OptionalBytesReference.of((BytesReference)BlobContainerUtils.getRegisterUsingConsistentRead((InputStream)stream, (String)container, (String)key));
            }
            return optionalBytesReference;
        }
        catch (Exception e) {
            int statusCode;
            BaseServiceException serviceException = GoogleCloudStorageBlobStore.unwrapServiceException(e);
            if (serviceException != null && (statusCode = serviceException.getCode()) == RestStatus.NOT_FOUND.getStatus()) {
                return OptionalBytesReference.EMPTY;
            }
            throw e;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    OptionalBytesReference compareAndExchangeRegister(String blobName, String container, String key, BytesReference expected, BytesReference updated) throws IOException {
        long generation;
        BlobContainerUtils.ensureValidRegisterContent((BytesReference)updated);
        BlobId blobId = BlobId.of((String)this.bucketName, (String)blobName);
        Blob blob = SocketAccess.doPrivilegedIOException(() -> this.client().get(blobId));
        if (blob == null || blob.getGeneration() == null) {
            if (expected.length() != 0) {
                return OptionalBytesReference.EMPTY;
            }
            generation = 0L;
        } else {
            generation = blob.getGeneration();
            try (PrivilegedReadChannelStream stream = new PrivilegedReadChannelStream((ReadableByteChannel)SocketAccess.doPrivilegedIOException(() -> this.client().reader(blobId, new Storage.BlobSourceOption[]{Storage.BlobSourceOption.generationMatch((long)generation)})));){
                BytesReference witness = BlobContainerUtils.getRegisterUsingConsistentRead((InputStream)stream, (String)container, (String)key);
                if (!witness.equals(expected)) {
                    OptionalBytesReference throwable = OptionalBytesReference.of((BytesReference)witness);
                    return throwable;
                }
            }
            catch (Exception e) {
                OptionalBytesReference optionalBytesReference;
                BaseServiceException serviceException = GoogleCloudStorageBlobStore.unwrapServiceException(e);
                if (serviceException == null) throw e;
                int statusCode = serviceException.getCode();
                if (statusCode != RestStatus.NOT_FOUND.getStatus()) {
                    if (statusCode != RestStatus.PRECONDITION_FAILED.getStatus()) throw e;
                    return OptionalBytesReference.MISSING;
                }
                if (expected.length() == 0) {
                    optionalBytesReference = OptionalBytesReference.MISSING;
                    return optionalBytesReference;
                }
                optionalBytesReference = OptionalBytesReference.EMPTY;
                return optionalBytesReference;
            }
        }
        BlobInfo blobInfo = BlobInfo.newBuilder((BlobId)BlobId.of((String)this.bucketName, (String)blobName, (Long)generation)).setMd5(Base64.getEncoder().encodeToString(MessageDigests.digest((BytesReference)updated, (MessageDigest)MessageDigests.md5()))).build();
        BytesRef bytesRef = updated.toBytesRef();
        try {
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)() -> this.client().create(blobInfo, bytesRef.bytes, bytesRef.offset, bytesRef.length, new Storage.BlobTargetOption[]{Storage.BlobTargetOption.generationMatch()})));
            return OptionalBytesReference.of((BytesReference)expected);
        }
        catch (Exception e) {
            BaseServiceException serviceException = GoogleCloudStorageBlobStore.unwrapServiceException(e);
            if (serviceException == null) throw e;
            int statusCode = serviceException.getCode();
            if (statusCode == RestStatus.PRECONDITION_FAILED.getStatus()) return OptionalBytesReference.MISSING;
            if (statusCode != RestStatus.TOO_MANY_REQUESTS.getStatus()) throw e;
            return OptionalBytesReference.MISSING;
        }
    }

    private static BaseServiceException unwrapServiceException(Throwable t) {
        for (int i = 0; i < 10 && t != null; t = t.getCause(), ++i) {
            if (!(t instanceof BaseServiceException)) continue;
            BaseServiceException baseServiceException = (BaseServiceException)t;
            return baseServiceException;
        }
        return null;
    }

    static {
        String key = "es.repository_gcs.large_blob_threshold_byte_size";
        String largeBlobThresholdByteSizeProperty = System.getProperty("es.repository_gcs.large_blob_threshold_byte_size");
        if (largeBlobThresholdByteSizeProperty == null) {
            LARGE_BLOB_THRESHOLD_BYTE_SIZE = Math.toIntExact(new ByteSizeValue(5L, ByteSizeUnit.MB).getBytes());
        } else {
            int largeBlobThresholdByteSize;
            try {
                largeBlobThresholdByteSize = Integer.parseInt(largeBlobThresholdByteSizeProperty);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("failed to parse es.repository_gcs.large_blob_threshold_byte_size having value [" + largeBlobThresholdByteSizeProperty + "]");
            }
            if (largeBlobThresholdByteSize <= 0) {
                throw new IllegalArgumentException("es.repository_gcs.large_blob_threshold_byte_size must be positive but was [" + largeBlobThresholdByteSizeProperty + "]");
            }
            LARGE_BLOB_THRESHOLD_BYTE_SIZE = largeBlobThresholdByteSize;
        }
        NO_OVERWRITE_NO_MD5 = new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist()};
        OVERWRITE_NO_MD5 = new Storage.BlobWriteOption[0];
        NO_OVERWRITE_CHECK_MD5 = new Storage.BlobWriteOption[]{Storage.BlobWriteOption.doesNotExist(), Storage.BlobWriteOption.md5Match()};
        OVERWRITE_CHECK_MD5 = new Storage.BlobWriteOption[]{Storage.BlobWriteOption.md5Match()};
    }

    private static final class WritableBlobChannel
    implements WritableByteChannel {
        private final WriteChannel channel;

        WritableBlobChannel(WriteChannel writeChannel) {
            this.channel = writeChannel;
        }

        @Override
        @SuppressForbidden(reason="channel is based on a socket")
        public int write(ByteBuffer src) throws IOException {
            return SocketAccess.doPrivilegedIOException(() -> this.channel.write(src));
        }

        @Override
        public boolean isOpen() {
            return this.channel.isOpen();
        }

        @Override
        public void close() {
        }
    }

    private static final class PrivilegedReadChannelStream
    extends InputStream {
        private final InputStream stream;

        PrivilegedReadChannelStream(ReadableByteChannel channel) {
            this.stream = Channels.newInputStream(channel);
        }

        @Override
        public int read(byte[] b) throws IOException {
            return SocketAccess.doPrivilegedIOException(() -> this.stream.read(b));
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            return SocketAccess.doPrivilegedIOException(() -> this.stream.read(b, off, len));
        }

        @Override
        public void close() throws IOException {
            SocketAccess.doPrivilegedVoidIOException((CheckedRunnable<IOException>)((CheckedRunnable)this.stream::close));
        }

        @Override
        public int read() throws IOException {
            return SocketAccess.doPrivilegedIOException(this.stream::read);
        }
    }
}

